import { Atom } from '@tldraw/state';
import { AtomOptions } from '@tldraw/state';
import { Computed } from '@tldraw/state';
import { ComputedOptions } from '@tldraw/state';
import { FunctionComponent } from 'react';
import { default as React_2 } from 'react';
import { Signal } from '@tldraw/state';

/**
 * Returns a tracked version of the given component.
 * Any signals whose values are read while the component renders will be tracked.
 * If any of the tracked signals change later it will cause the component to re-render.
 *
 * This also wraps the component in a React.memo() call, so it will only re-render if the props change.
 *
 * @example
 * ```ts
 * const Counter = track(function Counter(props: CounterProps) {
 *   const count = useAtom('count', 0)
 *   const increment = useCallback(() => count.set(count.get() + 1), [count])
 *   return <button onClick={increment}>{count.get()}</button>
 * })
 * ```
 *
 * @param baseComponent - The base component to track.
 * @public
 */
export declare function track<T extends FunctionComponent<any>>(baseComponent: T): React_2.NamedExoticComponent<React_2.ComponentProps<T>>;

/**
 * Creates a new atom and returns it. The atom will be created only once.
 *
 * See [[atom]]
 *
 * @example
 * ```ts
 * const Counter = track(function Counter () {
 *   const count = useAtom('count', 0)
 *   const increment = useCallback(() => count.set(count.get() + 1), [count])
 *   return <button onClick={increment}>{count.get()}</button>
 * })
 * ```
 *
 * @public
 */
export declare function useAtom<Value, Diff = unknown>(
/**
 * The name of the atom. This does not need to be globally unique. It is used for debugging and performance profiling.
 */
name: string, 
/**
 * The initial value of the atom. If this is a function, it will be called to get the initial value.
 */
valueOrInitialiser: (() => Value) | Value, 
/**
 * Options for the atom.
 */
options?: AtomOptions<Value, Diff>): Atom<Value, Diff>;

/**
 * Creates a new computed signal and returns it. The computed signal will be created only once.
 *
 * See [[computed]]
 *
 * @example
 * ```ts
 * type GreeterProps = {
 *   firstName: Signal<string>
 *   lastName: Signal<string>
 * }
 *
 * const Greeter = track(function Greeter ({firstName, lastName}: GreeterProps) {
 *   const fullName = useComputed('fullName', () => `${firstName.get()} ${lastName.get()}`)
 *   return <div>Hello {fullName.get()}!</div>
 * })
 * ```
 *
 * @public
 */
export declare function useComputed<Value>(name: string, compute: () => Value, deps: any[]): Computed<Value>;

/** @public */
export declare function useComputed<Value, Diff = unknown>(name: string, compute: () => Value, opts: ComputedOptions<Value, Diff>, deps: any[]): Computed<Value>;

/** @public */
export declare function useQuickReactor(name: string, reactFn: () => void, deps?: any[]): void;

/** @public */
export declare function useReactor(name: string, reactFn: () => void, deps?: any[] | undefined): void;

/**
 * Wraps some synchronous react render logic in a reactive tracking context.
 *
 * This allows you to use reactive values transparently.
 *
 * @example
 * ```ts
 * function MyComponent() {
 *   return useStateTracking('MyComponent', () => {
 *     const editor = useEditor()
 *     return <div>Num shapes: {editor.getCurrentPageShapes().length}</div>
 *   })
 * }
 * ```
 *
 * @see the `track` component wrapper, which uses this under the hood.
 *
 * @public
 */
export declare function useStateTracking<T>(name: string, render: () => T): T;

/**
 * Extracts the value from a signal and subscribes to it.
 *
 * Note that you do not need to use this hook if you are wrapping the component with [[track]]
 *
 * @example
 * ```ts
 * const Counter: React.FC = () => {
 *   const $count = useAtom('count', 0)
 *   const increment = useCallback(() => $count.set($count.get() + 1), [count])
 *   const currentCount = useValue($count)
 *   return <button onClick={increment}>{currentCount}</button>
 * }
 * ```
 *
 * You can also pass a function to compute the value and it will be memoized as in [[useComputed]]:
 *
 * @example
 * ```ts
 * type GreeterProps = {
 *   firstName: Signal<string>
 *   lastName: Signal<string>
 * }
 *
 * const Greeter = track(function Greeter({ firstName, lastName }: GreeterProps) {
 *   const fullName = useValue('fullName', () => `${firstName.get()} ${lastName.get()}`, [
 *     firstName,
 *     lastName,
 *   ])
 *   return <div>Hello {fullName}!</div>
 * })
 * ```
 *
 * @public
 */
export declare function useValue<Value>(value: Signal<Value>): Value;

/** @public */
export declare function useValue<Value>(name: string, fn: () => Value, deps: unknown[]): Value;

export { }
